<?php

namespace Drupal\Tests\nodeaccess\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\NodeType;
use Drupal\user\Entity\Role;

/**
 * Test permissions generated by permission_callbacks.
 *
 * @group nodeaccess
 */
class NodeGrantPermissionsTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['system', 'user', 'field'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
  }

  /**
   * Tests the removal of dynamic generated permissions.
   */
  public function testDynamicPermissions() {
    include_once $this->root . '/core/includes/install.core.inc';
    $version_info = _install_get_version_info(\Drupal::VERSION);
    $major_version = (int) $version_info['major'];
    $minor_version = (int) $version_info['minor'];
    if ($major_version < 9 || ($major_version === 9 && $minor_version < 3)) {
      $this->markTestSkipped('Permissions can define dependencies since 9.3.0');
    }
    $this->enableModules(['node', 'nodeaccess']);
    /** @var \Drupal\user\RoleStorage $role_storage */
    $role_storage = $this->container->get('entity_type.manager')->getStorage('user_role');
    /** @var \Drupal\user\RoleInterface $role */
    $role = Role::create([
      'id' => 'test_role',
      'label' => $this->randomString(),
    ]);
    $role->save();

    /** @var \Drupal\node\NodeTypeInterface $node_type */
    $node_type = NodeType::create([
      'type' => mb_strtolower($this->randomMachineName()),
      'name' => $this->randomString(),
    ]);
    $node_type->save();

    $permission_administer_nodeaccess = 'administer nodeaccess';
    $permission_grant_node = 'grant node permissions';
    $permission_grant_bundle = "nodeaccess grant {$node_type->id()} permissions";

    $role
      ->grantPermission($permission_administer_nodeaccess)
      ->grantPermission($permission_grant_node)
      ->grantPermission($permission_grant_bundle)
      ->save();

    // Make sure the role has the permissions granted.
    $role_storage->resetCache();
    $role = Role::load($role->id());
    $this->assertTrue($role->hasPermission($permission_administer_nodeaccess));
    $this->assertTrue($role->hasPermission($permission_grant_node));
    $this->assertTrue($role->hasPermission($permission_grant_bundle));

    // Deleted the node type, the corresponding dynamic generated bundle
    // permission(s) should be removed. In this case, it's
    // $permission_grant_bundle. But non-dynamic generated ones should be there.
    $node_type->delete();
    $role_storage->resetCache();
    $role = Role::load($role->id());
    $this->assertNotNull($role);
    $this->assertFalse($role->hasPermission($permission_grant_bundle));
    $this->assertTrue($role->hasPermission($permission_administer_nodeaccess));
    $this->assertTrue($role->hasPermission($permission_grant_node));

    // Trigger the removal of configuration belonging to the module manually
    // because KernelTestBase::disableModules() is not aware of this.
    $this->container->get('config.manager')->uninstall('module', 'nodeaccess');
    $this->disableModules(['nodeaccess']);

    // After uninstalling the module, all permissions defined by this module
    // should be revoked.
    $role_storage->resetCache();
    $role = Role::load($role->id());
    $this->assertNotNull($role);
    $this->assertFalse($role->hasPermission($permission_grant_bundle));
    $this->assertFalse($role->hasPermission($permission_administer_nodeaccess));
    $this->assertFalse($role->hasPermission($permission_grant_node));

  }

}
