<?php

namespace Drupal\custom_qr_code\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\node\Entity\Node;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class GenerateQRCodeForm extends FormBase {

    protected $entityTypeManager;

    public function __construct(EntityTypeManagerInterface $entityTypeManager) {
        $this->entityTypeManager = $entityTypeManager;
    }

    public static function create(ContainerInterface $container) {
        return new static(
        $container->get('entity_type.manager')
        );
    }

    public function getFormId() {
        return 'generate_qr_code_form';
    }

    public function buildForm(array $form, FormStateInterface $form_state) {
        // Field to select SPBU reference
        $form['field_spbu'] = [
            '#type' => 'entity_autocomplete',
            '#title' => $this->t('Select SPBU'),
            '#target_type' => 'node',
            '#tags' => TRUE,
            '#description' => $this->t('Select the SPBU content type.'),
            '#required' => TRUE,
            '#selection_handler' => 'default:node',
            '#selection_handler_settings' => [
                'target_bundles' => [
                    'spbu' => 'spbu', // Ensure this matches your content type's machine name
                ],
            ],
            '#entity_type' => 'node', // Add this line to specify the entity type explicitly
        ];

        $form['field_bbm'] = [
            '#type' => 'select',
            '#title' => $this->t('Select BBM'),
            '#options' => [
              'pertalite' => $this->t('Pertalite'),
              'pertamax' => $this->t('Pertamax'),
              'biosolar' => $this->t('BioSolar'),
              'dexlite' => $this->t('Dexlite'),
              'dex' => $this->t('Dex'),
            ],
            '#description' => $this->t('Please select an option.'),
            '#required' => TRUE,
          ];

        // Field to enter liter
        $form['field_liter'] = [
            '#type' => 'number',
            '#title' => $this->t('Liter'),
            '#description' => $this->t('Enter the liter for the vouchers.'),
            '#required' => TRUE,
        ];

        // Field to enter number of vouchers to create
        $form['field_how_many'] = [
            '#type' => 'number',
            '#title' => $this->t('Number of Vouchers'),
            '#description' => $this->t('Enter how many vouchers you want to create.'),
            '#required' => TRUE,
            '#min' => 1,
        ];

        // Submit button
        $form['submit'] = [
            '#type' => 'submit',
            '#value' => $this->t('Generate Vouchers'),
        ];

        return $form;
    }

    public function submitForm(array &$form, FormStateInterface $form_state) {
        // Get form values
        $spbu_id = $form_state->getValue('field_spbu');
        $liter = $form_state->getValue('field_liter');
        $how_many = $form_state->getValue('field_how_many');
        $bbm = $form_state->getValue('field_bbm');

        
        // Load SPBU node
        $spbu_node = Node::load($spbu_id[0]['target_id']);

        if ($spbu_node) {
            for ($i = 0; $i < $how_many; $i++) {
                $qr_code = $this->generateUniqueQrCode();
                $url = "http://conan.goteku.com:8080/voucher/" . $qr_code;

                // Create a new voucher node
                $voucher = Node::create([
                    'type' => 'voucher_spbu',
                    'field_spbu' => $spbu_id, 
                    'field_bbm' => $bbm, 
                    'field_liter' => $liter, 
                    'title' => $qr_code,
                    'field_qr_code' => $url, 
                ]);

                // Save the node
                $voucher->save();
            }

            \Drupal::messenger()->addMessage($this->t('@count vouchers created.', ['@count' => $how_many]));
        } else {
            \Drupal::messenger()->addError($this->t('SPBU node not found.'));
        }
    }

    private function generateUniqueQrCode() {
        $length = 10;
        $unique = false;
        $qr_code = '';
    
        while (!$unique) {
            $qr_code = date('Ymd') . substr(str_shuffle(str_repeat('0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ', ceil($length / 62))), 1, $length);
            
    
            // Check if the generated QR code already exists in the database.
            $exists = \Drupal::entityQuery('node')
                ->condition('field_qr_code', $qr_code) // Replace with your field name.
                ->accessCheck(FALSE) // Add this line to bypass access checks
                ->count()
                ->execute();
    
            if ($exists == 0) {
                $unique = true; // Unique QR code found.
            }
        }
    
        return $qr_code;
    }
}